"""Analyzer agent.

Responsibilities:
  * compute team strengths (attack/defense multipliers)
  * provide simple helper views (form, head-to-head) for API/LLM

Strengths model
--------------
Bookmakers and professional models usually rely on **recent windows** of games
(10–30), with heavier weight on the most recent matches. We implement:
  - configurable window (default 20)
  - recency weights: last 5 games weigh more
  - minimum sample guard
  - normalization around league average goals
"""

from __future__ import annotations

import time
from typing import Any

from database.manager import DatabaseManager


class AnalyzerAgent:
    def __init__(self, db: DatabaseManager, window: int = 20):
        self.db = db
        self.window = int(window)

    async def team_form(self, team: str, n: int = 5) -> list[dict[str, Any]]:
        matches = await self.db.last_matches_for_team(team, limit=max(5, n))
        out: list[dict[str, Any]] = []
        for m in matches[:n]:
            if m.home_score is None or m.away_score is None:
                continue
            is_home = (m.home_team == team)
            gf = m.home_score if is_home else m.away_score
            ga = m.away_score if is_home else m.home_score
            res = "W" if gf > ga else "D" if gf == ga else "L"
            out.append({"match_id": m.id, "gf": gf, "ga": ga, "res": res, "kickoff_ts": m.kickoff_ts})
        return out

    async def h2h(self, a: str, b: str, limit: int = 8) -> list[dict[str, Any]]:
        matches = await self.db.head_to_head(a, b, limit=limit)
        out = []
        for m in matches:
            out.append(
                {
                    "match_id": m.id,
                    "kickoff_ts": m.kickoff_ts,
                    "home": m.home_team,
                    "away": m.away_team,
                    "score": [m.home_score, m.away_score],
                }
            )
        return out

    async def recompute_team_strengths(self) -> int:
        """Compute attack/defense multipliers and upsert into `team_stats`.

        Returns number of updated teams.

        IMPORTANT: This requires finished matches with scores.
        """

        # NOTE: We use SQL because it's significantly faster for windowing than
        # pulling all matches into Python.
        sql = f"""
        WITH team_games AS (
            SELECT
                home_team AS team,
                home_score AS goals_for,
                away_score AS goals_against,
                COALESCE(kickoff_ts, last_update_ts) AS ts
            FROM matches
            WHERE status='finished'
              AND home_score IS NOT NULL AND away_score IS NOT NULL

            UNION ALL

            SELECT
                away_team AS team,
                away_score AS goals_for,
                home_score AS goals_against,
                COALESCE(kickoff_ts, last_update_ts) AS ts
            FROM matches
            WHERE status='finished'
              AND home_score IS NOT NULL AND away_score IS NOT NULL
        ),
        ranked AS (
            SELECT
                team,
                goals_for,
                goals_against,
                ts,
                ROW_NUMBER() OVER (PARTITION BY team ORDER BY ts DESC NULLS LAST) AS rn
            FROM team_games
        ),
        windowed AS (
            SELECT
                team,
                goals_for,
                goals_against,
                rn,
                CASE
                    WHEN rn <= 5 THEN 1.5
                    WHEN rn <= 15 THEN 1.0
                    WHEN rn <= {self.window} THEN 0.7
                    ELSE 0
                END AS w
            FROM ranked
            WHERE rn <= {self.window}
        ),
        agg AS (
            SELECT
                team,
                SUM(goals_for * w) / NULLIF(SUM(w),0) AS gf,
                SUM(goals_against * w) / NULLIF(SUM(w),0) AS ga,
                COUNT(*) AS n
            FROM windowed
            GROUP BY team
            HAVING COUNT(*) >= 5
        ),
        league AS (
            SELECT
                AVG(gf) AS league_gf,
                AVG(ga) AS league_ga
            FROM agg
        )
        SELECT
            a.team AS team,
            a.gf AS gf,
            a.ga AS ga,
            a.n  AS n,
            l.league_gf AS league_gf,
            l.league_ga AS league_ga
        FROM agg a
        CROSS JOIN league l
        """

        rows = await self.db.fetch_all(sql)
        if not rows:
            return 0

        now = int(time.time())
        updated = 0

        # Convert to multipliers around league average.
        for r in rows:
            team = str(r["team"])
            gf = float(r["gf"] or 0.0)
            ga = float(r["ga"] or 0.0)
            league_gf = float(r["league_gf"] or 1.35)
            league_ga = float(r["league_ga"] or 1.35)

            # Attack multiplier: >1 means scores more than league.
            attack = gf / max(0.2, league_gf)
            # Defense multiplier: >1 means concedes more than league (worse).
            defense = ga / max(0.2, league_ga)

            # Bound to sane range.
            attack = max(0.65, min(attack, 1.65))
            defense = max(0.65, min(defense, 1.65))

            await self.db.upsert_team_stats(team_id=team, team_name=team, attack=attack, defense=defense, updated_ts=now)
            updated += 1

        return updated
