"""Predictor agent (Poisson + team strength + basic form).

The original issue "all прогнозы одинаковые" happened because lambdas
(`exp_home_goals`/`exp_away_goals`) were constant fallback numbers.

We fix that by:
  1) computing team attack/defense multipliers (AnalyzerAgent -> team_stats)
  2) using those multipliers to compute different lambdas per match
  3) deriving a *calibrated-ish* confidence score based on input quality

This module is deterministic and does not require OpenAI. LLM insights are served separately.
"""

from __future__ import annotations

import math
import time
from dataclasses import dataclass

from database.manager import DatabaseManager


@dataclass
class _PoissonProbs:
    p1: float
    px: float
    p2: float


def _poisson_pmf(k: int, lam: float) -> float:
    return math.exp(-lam) * (lam ** k) / math.factorial(k)


def _match_outcome_probs(lambda_home: float, lambda_away: float, max_goals: int = 8) -> _PoissonProbs:
    # compute joint distribution up to max_goals
    p_home = 0.0
    p_draw = 0.0
    p_away = 0.0
    ph = [_poisson_pmf(i, lambda_home) for i in range(max_goals + 1)]
    pa = [_poisson_pmf(i, lambda_away) for i in range(max_goals + 1)]
    for i in range(max_goals + 1):
        for j in range(max_goals + 1):
            p = ph[i] * pa[j]
            if i > j:
                p_home += p
            elif i == j:
                p_draw += p
            else:
                p_away += p
    # normalize (tail mass beyond max_goals)
    s = p_home + p_draw + p_away
    if s <= 0:
        return _PoissonProbs(1 / 3, 1 / 3, 1 / 3)
    return _PoissonProbs(p_home / s, p_draw / s, p_away / s)


class PredictorAgent:
    def __init__(self, db: DatabaseManager):
        self.db = db

    async def predict_for_match(self, match) -> dict:
        """Return prediction dict for `database.models.Match`."""

        # Base league average goals per team per game (rough EPL-ish).
        base = 1.35
        home_adv = 1.12

        home_stats = await self.db.get_team_stats(match.home_team)
        away_stats = await self.db.get_team_stats(match.away_team)

        if home_stats and away_stats:
            # defense multiplier >1 means *worse* defense (concedes more)
            lambda_home = base * float(home_stats.attack) * float(away_stats.defense) * home_adv
            lambda_away = base * float(away_stats.attack) * float(home_stats.defense)
            quality = 1.0
        else:
            # fallback: still slightly different via home advantage only
            lambda_home = base * home_adv
            lambda_away = base
            quality = 0.35

        # Clamp to sane range
        lambda_home = max(0.2, min(lambda_home, 3.5))
        lambda_away = max(0.2, min(lambda_away, 3.5))

        probs = _match_outcome_probs(lambda_home, lambda_away, max_goals=8)

        # Confidence: combine "sharpness" (how far from uniform) with input quality.
        # Sharpness is 0 when probs ~ uniform, increases with max prob.
        sharp = max(probs.p1, probs.px, probs.p2) - (1.0 / 3.0)
        sharp = max(0.0, min(sharp / (2.0 / 3.0), 1.0))  # normalize to 0..1
        confidence = 0.05 + 0.85 * (0.65 * sharp + 0.35 * quality)
        confidence = max(0.01, min(confidence, 0.99))

        now = int(time.time())
        return {
            "match_id": match.id,
            "model": "poisson+form",
            "p_home_win": float(probs.p1),
            "p_draw": float(probs.px),
            "p_away_win": float(probs.p2),
            "exp_home_goals": float(lambda_home),
            "exp_away_goals": float(lambda_away),
            "confidence": float(confidence),
            "created_ts": now,
        }