from __future__ import annotations
import json
from typing import Any, Optional
from openai import OpenAI
from config import settings

_client: Optional[OpenAI] = None


def client() -> Optional[OpenAI]:
    global _client
    if not settings.OPENAI_API_KEY:
        return None
    if _client is None:
        _client = OpenAI(api_key=settings.OPENAI_API_KEY)
    return _client


def build_prompt(ctx: dict[str, Any]) -> str:
    return (
        "Ты — футбольный аналитик. Дай инсайты по матчу на основе данных.\n"
        "Верни СТРОГО JSON следующего вида:\n"
        "{\n"
        '  "summary": string,\n'
        '  "key_factors": string[],\n'
        '  "risks": string[],\n'
        '  "suggested_angles": string[],\n'
        '  "data_gaps": string[]\n'
        "}\n"
        "Данные:\n"
        + json.dumps(ctx, ensure_ascii=False)
    )


def get_insight(ctx: dict[str, Any]) -> Optional[dict[str, Any]]:
    c = client()
    if not c:
        return None

    r = c.responses.create(
        model=settings.OPENAI_MODEL,
        input=build_prompt(ctx),
        max_output_tokens=700,
    )

    text = getattr(r, "output_text", "") or ""

    try:
        return json.loads(text)
    except Exception:
        # fallback: пытаемся вытащить JSON из текста
        start = text.find("{")
        end = text.rfind("}")
        if start != -1 and end != -1 and end > start:
            try:
                return json.loads(text[start : end + 1])
            except Exception:
                pass

        return {
            "summary": text[:1500],
            "key_factors": [],
            "risks": ["unstructured_output"],
            "suggested_angles": [],
            "data_gaps": [],
        }

