from __future__ import annotations
from pydantic_settings import BaseSettings
from pydantic import Field

class Settings(BaseSettings):
    ENV: str = "prod"

    DATABASE_URL: str

    OPENAI_API_KEY: str = ""
    OPENAI_MODEL: str = "gpt-4o-mini"

    TELEGRAM_BOT_TOKEN: str
    TELEGRAM_ADMIN_CHAT_ID: int = 0

    # Multiple competition URLs separated by '|'
    FLASHSCORE_COMPETITION_URLS: str = "https://www.flashscore.com/football/"

    PARSER_USE_PLAYWRIGHT: bool = True
    PARSER_HEADLESS: bool = True
    PARSER_MIN_DELAY_S: float = 1.2
    PARSER_CACHE_TTL_S: int = 45

    LOOP_TICK_S: int = 15
    REFRESH_UPCOMING_EVERY_S: int = 120
    REFRESH_LIVE_EVERY_S: int = 20

    # If True, worker will also query "results" pages for each competition URL
    # to build enough finished matches for team-strength computation.
    INCLUDE_RESULTS_PAGES: bool = True

    ENABLE_GOAL_NOTIFICATIONS: bool = True

    def competition_urls(self) -> list[str]:
        raw = self.FLASHSCORE_COMPETITION_URLS.strip()
        if not raw:
            return []
        urls = [u.strip().rstrip("/") + "/" for u in raw.split("|") if u.strip()]
        if not self.INCLUDE_RESULTS_PAGES:
            return urls

        # Add results pages next to the main competition page if not provided.
        expanded: list[str] = []
        for u in urls:
            expanded.append(u)
            if "results" not in u:
                expanded.append(u.rstrip("/") + "/results/")
        # Dedup preserving order
        seen = set()
        out = []
        for u in expanded:
            if u in seen:
                continue
            seen.add(u)
            out.append(u)
        return out

    class Config:
        env_file = ".env"
        case_sensitive = False

settings = Settings()
