"""Database bootstrap and SQLAlchemy async engine.

We intentionally use a single DB stack across the project:

* SQLAlchemy 2.x async engine
* asyncpg driver
* ORM models (see :mod:`database.models`)

Earlier iterations mixed raw `asyncpg` DDL with ORM access which led to tables
existing but columns not matching what the code expected (and then nothing was
inserted). This module prevents that.
"""

from __future__ import annotations

from sqlalchemy.ext.asyncio import AsyncEngine, async_sessionmaker, create_async_engine

from config import settings
from database.models import Base


def _normalize_database_url(url: str) -> str:
    """Accept both `postgresql://` and `postgresql+asyncpg://` styles."""
    url = url.strip()
    if url.startswith("postgresql://"):
        # SQLAlchemy async requires the explicit asyncpg driver.
        return url.replace("postgresql://", "postgresql+asyncpg://", 1)
    return url


DATABASE_URL = _normalize_database_url(settings.DATABASE_URL)


engine: AsyncEngine = create_async_engine(
    DATABASE_URL,
    pool_pre_ping=True,
    pool_size=5,
    max_overflow=10,
)


SessionLocal = async_sessionmaker(
    bind=engine,
    expire_on_commit=False,
    autoflush=False,
    autocommit=False,
)


async def init_db() -> None:
    """Create DB schema if it doesn't exist."""
    async with engine.begin() as conn:
        await conn.run_sync(Base.metadata.create_all)
