from __future__ import annotations
import asyncio, time, json, logging, os
from aiogram import Bot, Dispatcher, F
from aiogram.types import Message
from aiogram.filters import Command
from config import settings
from database.manager import DatabaseManager

log = logging.getLogger("bot")
logging.basicConfig(level=logging.INFO, format="%(asctime)s | %(levelname)s | %(name)s | %(message)s")

db = DatabaseManager()

def fmt_match(m) -> str:
    score = ""
    if m.home_score is not None and m.away_score is not None:
        score = f" {m.home_score}:{m.away_score}"
    minute = f" ({m.minute}')" if m.minute else ""
    return f"{m.home_team} — {m.away_team}{score}{minute} [{m.status}]\n{m.match_url}"

async def cmd_today(message: Message):
    rows = await db.list_matches(limit=20)
    text = "\n\n".join(fmt_match(r) for r in rows[:20]) or "Нет матчей в БД"
    await message.answer(text[:4000])

async def cmd_live(message: Message):
    rows = await db.list_matches(status="live", limit=30)
    text = "\n\n".join(fmt_match(r) for r in rows) or "Сейчас нет live-матчей"
    await message.answer(text[:4000])

async def cmd_predict(message: Message):
    rows = await db.list_matches(limit=10)
    out = []
    for m in rows[:10]:
        p = await db.get_prediction_for_match(m.id, model="poisson+form")
        if not p:
            continue
        out.append(
            f"{m.home_team} — {m.away_team}\n"
            f"P1={p.p_home_win:.2f} X={p.p_draw:.2f} P2={p.p_away_win:.2f} conf={p.confidence:.2f}\n"
            f"EG={p.exp_home_goals:.2f}:{p.exp_away_goals:.2f}\n{m.match_url}"
        )
    await message.answer("\n\n".join(out)[:4000] if out else "Нет прогнозов в БД")

async def cmd_metrics(message: Message):
    agg = await db.get_metrics(model="poisson+form")
    if not agg:
        await message.answer("Метрик пока нет. Дай воркеру накопить finished-матчи.")
        return
    await message.answer(
        f"Model: {agg.model}\n"
        f"Matches: {agg.n_matches}\n"
        f"Accuracy: {agg.accuracy:.3f}\n"
        f"Brier: {agg.brier:.3f}\n"
        f"LogLoss: {agg.logloss:.3f}\n"
        f"Updated: {agg.updated_ts}"
    )

async def notify_loop(bot: Bot):
    if not settings.ENABLE_GOAL_NOTIFICATIONS:
        log.info("Goal notifications disabled")
        return
    admin = settings.TELEGRAM_ADMIN_CHAT_ID
    if admin == 0:
        log.info("TELEGRAM_ADMIN_CHAT_ID=0; notifications will not be sent")
        return

    last_ts = int(time.time()) - 60
    log.info("Notification loop started to chat_id=%s", admin)

    while True:
        try:
            events = await db.list_events_since(last_ts, limit=200)
            for ev in events:
                last_ts = max(last_ts, ev.created_ts)
                if ev.event_type != "match_update":
                    continue
                payload = json.loads(ev.payload_json or "{}")
                # goal detection: score change
                if "home_score" in payload or "away_score" in payload:
                    msg = f"⚽️ Update {ev.match_id}: {json.dumps(payload, ensure_ascii=False)}"
                    await bot.send_message(admin, msg[:4000])
        except Exception as e:
            log.exception("notify loop error: %s", e)
        await asyncio.sleep(5)

async def main():
    bot = Bot(token=settings.TELEGRAM_BOT_TOKEN)
    dp = Dispatcher()

    dp.message.register(cmd_today, Command("today"))
    dp.message.register(cmd_live, Command("live"))
    dp.message.register(cmd_predict, Command("predict"))
    dp.message.register(cmd_metrics, Command("metrics"))

    # background notify loop
    asyncio.create_task(notify_loop(bot))

    log.info("Bot started")
    await dp.start_polling(bot)

if __name__ == "__main__":
    asyncio.run(main())
