from __future__ import annotations
import time, json
from sqlalchemy import select, update, delete
from sqlalchemy.ext.asyncio import AsyncSession
from database.db import SessionLocal
from database.models import Match, Prediction, MetricAggregate, MatchEvent

class DatabaseManager:
    async def upsert_match(self, m: dict) -> Match:
        async with SessionLocal() as s:
            obj = await s.get(Match, m["id"])
            now = int(time.time())
            if not obj:
                obj = Match(
                    id=m["id"],
                    source=m.get("source","flashscore"),
                    competition_url=m.get("competition_url",""),
                    match_url=m.get("match_url",""),
                    kickoff_ts=m.get("kickoff_ts"),
                    status=m.get("status","unknown"),
                    home_team=m["home_team"],
                    away_team=m["away_team"],
                    minute=m.get("minute"),
                    home_score=m.get("home_score"),
                    away_score=m.get("away_score"),
                    last_seen_ts=now,
                    last_update_ts=now,
                )
                s.add(obj)
            else:
                # detect changes for events
                changed = {}
                for k in ["status","minute","home_score","away_score","kickoff_ts","match_url","competition_url"]:
                    if k in m and getattr(obj, k) != m[k]:
                        changed[k] = {"from": getattr(obj,k), "to": m[k]}
                        setattr(obj, k, m[k])
                obj.last_seen_ts = now
                if changed:
                    obj.last_update_ts = now
                    await self._insert_event(s, obj.id, "match_update", changed, now)
            await s.commit()
            return obj

    async def _insert_event(self, s: AsyncSession, match_id: str, event_type: str, payload: dict, ts: int) -> None:
        s.add(MatchEvent(match_id=match_id, event_type=event_type, payload_json=json.dumps(payload, ensure_ascii=False), created_ts=ts))

    async def list_matches(self, status: str | None = None, limit: int = 100):
        async with SessionLocal() as s:
            q = select(Match).order_by(Match.kickoff_ts.desc().nullslast()).limit(limit)
            if status:
                q = q.where(Match.status == status)
            res = await s.execute(q)
            return list(res.scalars().all())

    async def get_match(self, match_id: str) -> Match | None:
        async with SessionLocal() as s:
            return await s.get(Match, match_id)

    async def upsert_prediction(self, p: dict) -> None:
        async with SessionLocal() as s:
            # find by (match_id, model)
            q = select(Prediction).where(Prediction.match_id == p["match_id"], Prediction.model == p["model"])
            res = await s.execute(q)
            obj = res.scalars().first()
            now = int(time.time())
            if not obj:
                obj = Prediction(
                    match_id=p["match_id"],
                    model=p["model"],
                    p_home_win=p["p_home_win"],
                    p_draw=p["p_draw"],
                    p_away_win=p["p_away_win"],
                    exp_home_goals=p["exp_home_goals"],
                    exp_away_goals=p["exp_away_goals"],
                    confidence=p["confidence"],
                    created_ts=now,
                )
                s.add(obj)
            else:
                obj.p_home_win = p["p_home_win"]
                obj.p_draw = p["p_draw"]
                obj.p_away_win = p["p_away_win"]
                obj.exp_home_goals = p["exp_home_goals"]
                obj.exp_away_goals = p["exp_away_goals"]
                obj.confidence = p["confidence"]
            await s.commit()

    async def list_predictions(self, limit: int = 200):
        async with SessionLocal() as s:
            q = select(Prediction).order_by(Prediction.created_ts.desc()).limit(limit)
            res = await s.execute(q)
            return list(res.scalars().all())

    async def get_prediction_for_match(self, match_id: str, model: str = "poisson+form") -> Prediction | None:
        async with SessionLocal() as s:
            q = select(Prediction).where(Prediction.match_id == match_id, Prediction.model == model).limit(1)
            res = await s.execute(q)
            return res.scalars().first()

    async def last_matches_for_team(self, team: str, limit: int = 25):
        async with SessionLocal() as s:
            q = (
                select(Match)
                .where(
                    (Match.home_team == team) | (Match.away_team == team),
                    Match.status == "finished",
                    Match.home_score.is_not(None),
                    Match.away_score.is_not(None),
                )
                .order_by(Match.kickoff_ts.desc().nullslast())
                .limit(limit)
            )
            res = await s.execute(q)
            return list(res.scalars().all())

    async def head_to_head(self, a: str, b: str, limit: int = 12):
        async with SessionLocal() as s:
            q = (
                select(Match)
                .where(
                    ((Match.home_team == a) & (Match.away_team == b)) |
                    ((Match.home_team == b) & (Match.away_team == a)),
                    Match.status == "finished",
                    Match.home_score.is_not(None),
                    Match.away_score.is_not(None),
                )
                .order_by(Match.kickoff_ts.desc().nullslast())
                .limit(limit)
            )
            res = await s.execute(q)
            return list(res.scalars().all())

    async def save_metrics(self, model: str, n: int, acc: float, brier: float, logloss: float) -> None:
        async with SessionLocal() as s:
            q = select(MetricAggregate).where(MetricAggregate.model == model).limit(1)
            res = await s.execute(q)
            obj = res.scalars().first()
            now = int(time.time())
            if not obj:
                obj = MetricAggregate(model=model, n_matches=n, accuracy=acc, brier=brier, logloss=logloss, updated_ts=now)
                s.add(obj)
            else:
                obj.n_matches = n
                obj.accuracy = acc
                obj.brier = brier
                obj.logloss = logloss
                obj.updated_ts = now
            await s.commit()

    async def get_metrics(self, model: str = "poisson+form") -> MetricAggregate | None:
        async with SessionLocal() as s:
            q = select(MetricAggregate).where(MetricAggregate.model == model).limit(1)
            res = await s.execute(q)
            return res.scalars().first()

    async def list_events_since(self, since_ts: int, limit: int = 200):
        async with SessionLocal() as s:
            q = select(MatchEvent).where(MatchEvent.created_ts >= since_ts).order_by(MatchEvent.created_ts.asc()).limit(limit)
            res = await s.execute(q)
            return list(res.scalars().all())
